/*  
 * Copyright (C) 2005 Martin Pischky (mailto:martin@pischky.de)  
 *
 * This file (HexFileWriter.java) is part of IntelHexFile.
 *
 * IntelHexFile is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 */

/*
 * $Log: HexFileWriter.java,v $
 * Revision 1.1  2005/03/27 12:29:58  pischky
 * added
 *
 */

package org.fremo.ihx;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Writer;

/**
 * @author martin
 */
public class HexFileWriter {

    public static final int MAX_BYTES = 16;
    
    private Writer writer;
    
    /**
     * 
     */
    public HexFileWriter(Writer writer) {
        super();
        this.setWriter(writer);
    }

    /**
     * 
     */
    public HexFileWriter() {
        super();
    }

    /**
     * @throws IOException
     */
    public void close() 
    throws IOException 
    {
        this.getWriter().close();
    }
    
    public void write( Memory mem )
    throws IOException 
    {
        BufferedWriter w;
        if( getWriter() instanceof BufferedWriter ) {
            w = (BufferedWriter) this.getWriter();
        } else {
            w = new BufferedWriter(this.getWriter());
        }
        HexFileWriter.writeIntelHexFile(mem, w);
        HexFileWriter.writeEOF(w);
        if( ! (this.getWriter() instanceof BufferedWriter) ) {
            w.close();
        }
    }
    
    /**
     * @param writer
     * @throws IOException
     */
    private static void writeEOF( BufferedWriter writer ) 
    throws IOException 
    {
        writer.write( formatHexFileLine(HexFileLine.EOF) );
        writer.newLine();
    }
    
    /**
     * @param mem
     * @param writer
     * @throws IOException
     */
    public static void writeIntelHexFile( Memory mem, BufferedWriter writer ) 
    throws IOException 
    {
        int a = 0;
        while( a < Memory.ADDRESS_HIGH ) {
            a = mem.nextUsedAddress(a);
            if( a == -1 ) {
                break;
            }
            int cnt = 0;
            while( a+cnt <= Memory.ADDRESS_HIGH 
                   && mem.isDefined(a+cnt) 
                   && cnt < MAX_BYTES ) 
            {
                cnt++;
            }
            HexFileLine hl = new HexFileLine();
            hl.setRecordLength( cnt );
            hl.setAddress(a);
            hl.setRecordType(HexFileLine.DATA_RECORD);
            for (int i = 0; i < cnt; i++) {
                hl.setData(i, (byte) mem.get(a+i) );
            }
            hl.setCheckSum();
            writer.write( formatHexFileLine(hl) );
            writer.newLine();
            a += cnt;
        } 
    }
    
    private static String formatHexFileLine( HexFileLine hl ) {
        StringBuffer b = new StringBuffer(64);
        b.append(':');
        b.append( HexUtils.hexString2( hl.getRecordLength() ) );
        b.append( HexUtils.hexString4( hl.getAddress() ) );
        b.append( HexUtils.hexString2( hl.getRecordType() ) );
        for (int i = 0; i < hl.getRecordLength(); i++) {
            b.append( HexUtils.hexString2(hl.getData(i)) );
        }
        b.append( HexUtils.hexString2( hl.getCheckSum() ) );
        return b.toString();
    }

    /**
     * @param writer The writer to set.
     */
    public void setWriter(Writer writer) {
        this.writer = writer;
    }

    /**
     * @return Returns the writer.
     */
    public Writer getWriter() {
        return this.writer;
    }
    
}
